#include <errno.h>  // errno
#include <float.h>  // DBL_MIN, DBL_MAX
#include <math.h>   // INFINITY
#include <stdio.h>  // required for readline/readline.h (man readline)

#include "_build/detected-cpp-config.h"
#include "mycpp/runtime.h"
#ifdef HAVE_READLINE
  #include "cpp/frontend_pyreadline.h"
#endif

// Translation of Python's print().
void print(BigStr* s) {
  fputs(s->data_, stdout);  // print until first NUL
  fputc('\n', stdout);
}

BigStr* str(int i) {
  BigStr* s = OverAllocatedStr(kIntBufSize);
  int length = snprintf(s->data(), kIntBufSize, "%d", i);
  s->MaybeShrink(length);
  return s;
}

// TODO:
// - This could use a fancy exact algorithm, not libc
// - Does libc depend on locale?
BigStr* str(double d) {
  char buf[64];  // overestimate, but we use snprintf() to be safe

  // Problem:
  // %f prints 3.0000000 and 3.500000
  // %g prints 3 and 3.5
  //
  // We want literal syntax to indicate float, so add '.'

  int n = sizeof(buf) - 2;  // in case we add '.0'

  // %.9g digits for string that can be converted back to the same FLOAT
  // (not double)
  // https://stackoverflow.com/a/21162120
  // https://en.cppreference.com/w/cpp/types/numeric_limits/max_digits10
  int length = snprintf(buf, n, "%.9g", d);

  // %a is a hexfloat form, could use that somewhere
  // int length = snprintf(buf, n, "%a", d);

  if (strchr(buf, 'i')) {  // inf or -inf
    return StrFromC(buf);
  }

  if (!strchr(buf, '.')) {  // 12345 -> 12345.0
    buf[length] = '.';
    buf[length + 1] = '0';
    buf[length + 2] = '\0';
  }

  return StrFromC(buf);
}

// Do we need this API?  Or is mylib.InternedStr(BigStr* s, int start, int end)
// better for getting values out of Token.line without allocating?
//
// e.g. mylib.InternedStr(tok.line, tok.start, tok.start+1)
//
// Also for SmallStr, we don't care about interning.  Only for HeapStr.

BigStr* intern(BigStr* s) {
  // TODO: put in table gHeap.interned_
  return s;
}

// Print quoted string.  Called by StrFormat('%r').
// TODO: consider using J8 notation instead, since error messages show that
// string.
BigStr* repr(BigStr* s) {
  // Worst case: \0 becomes 4 bytes as '\\x00', and then two quote bytes.
  int n = len(s);
  int upper_bound = n * 4 + 2;

  BigStr* result = OverAllocatedStr(upper_bound);

  // Single quote by default.
  char quote = '\'';
  if (memchr(s->data_, '\'', n) && !memchr(s->data_, '"', n)) {
    quote = '"';
  }
  char* p = result->data_;

  // From PyString_Repr()
  *p++ = quote;
  for (int i = 0; i < n; ++i) {
    unsigned char c = static_cast<unsigned char>(s->data_[i]);
    if (c == quote || c == '\\') {
      *p++ = '\\';
      *p++ = c;
    } else if (c == '\t') {
      *p++ = '\\';
      *p++ = 't';
    } else if (c == '\n') {
      *p++ = '\\';
      *p++ = 'n';
    } else if (c == '\r') {
      *p++ = '\\';
      *p++ = 'r';
    } else if (0x20 <= c && c < 0x80) {
      *p++ = c;
    } else {
      // Unprintable becomes \xff.
      // TODO: Consider \yff.  This is similar to J8 strings, but we don't
      // decode UTF-8.
      sprintf(p, "\\x%02x", c & 0xff);
      p += 4;
    }
  }
  *p++ = quote;
  *p = '\0';

  int length = p - result->data_;
  result->MaybeShrink(length);
  return result;
}

// Helper functions that don't use exceptions.

bool StringToInt(const char* s, int length, int base, int* result) {
  if (length == 0) {
    return false;  // empty string isn't a valid integer
  }

  // Note: sizeof(int) is often 4 bytes on both 32-bit and 64-bit
  //       sizeof(long) is often 4 bytes on both 32-bit but 8 bytes on 64-bit
  // static_assert(sizeof(long) == 8);

  char* pos;  // mutated by strtol

  errno = 0;
  long v = strtol(s, &pos, base);

  if (errno == ERANGE) {
    switch (v) {
    case LONG_MIN:
      return false;  // underflow of long, which may be 64 bits
    case LONG_MAX:
      return false;  // overflow of long
    }
  }

  // It should ALSO fit in an int, not just a long
  if (v > INT_MAX) {
    return false;
  }
  if (v < INT_MIN) {
    return false;
  }

  const char* end = s + length;
  if (pos == end) {
    *result = v;
    return true;  // strtol() consumed ALL characters.
  }

  while (pos < end) {
    if (!IsAsciiWhitespace(*pos)) {
      return false;  // Trailing non-space
    }
    pos++;
  }

  *result = v;
  return true;  // Trailing space is OK
}

bool StringToInt64(const char* s, int length, int base, int64_t* result) {
  if (length == 0) {
    return false;  // empty string isn't a valid integer
  }

  // These should be the same type
  static_assert(sizeof(long long) == sizeof(int64_t));

  char* pos;  // mutated by strtol

  errno = 0;
  long long v = strtoll(s, &pos, base);

  if (errno == ERANGE) {
    switch (v) {
    case LLONG_MIN:
      return false;  // underflow
    case LLONG_MAX:
      return false;  // overflow
    }
  }

  const char* end = s + length;
  if (pos == end) {
    *result = v;
    return true;  // strtol() consumed ALL characters.
  }

  while (pos < end) {
    if (!IsAsciiWhitespace(*pos)) {
      return false;  // Trailing non-space
    }
    pos++;
  }

  *result = v;
  return true;  // Trailing space is OK
}

int to_int(BigStr* s, int base) {
  int i;
  if (StringToInt(s->data_, len(s), base, &i)) {
    return i;  // truncated to int
  } else {
    throw Alloc<ValueError>();
  }
}

BigStr* chr(int i) {
  // NOTE: i should be less than 256, in which we could return an object from
  // GLOBAL_STR() pool, like StrIter
  auto result = NewStr(1);
  result->data_[0] = i;
  return result;
}

int ord(BigStr* s) {
  assert(len(s) == 1);
  // signed to unsigned conversion, so we don't get values like -127
  uint8_t c = static_cast<uint8_t>(s->data_[0]);
  return c;
}

bool to_bool(BigStr* s) {
  return len(s) != 0;
}

double to_float(int i) {
  return static_cast<double>(i);
}

double to_float(BigStr* s) {
  char* begin = s->data_;
  char* end = begin + len(s);

  errno = 0;
  double result = strtod(begin, &end);

  if (errno == ERANGE) {  // error: overflow or underflow
    if (result >= HUGE_VAL) {
      return INFINITY;
    } else if (result <= -HUGE_VAL) {
      return -INFINITY;
    } else if (-DBL_MIN <= result && result <= DBL_MIN) {
      return 0.0;
    } else {
      FAIL("Invalid value after ERANGE");
    }
  }
  if (end == begin) {  // error: not a floating point number
    throw Alloc<ValueError>();
  }

  return result;
}

// e.g. ('a' in 'abc')
bool str_contains(BigStr* haystack, BigStr* needle) {
  // Common case
  if (len(needle) == 1) {
    return memchr(haystack->data_, needle->data_[0], len(haystack));
  }

  if (len(needle) > len(haystack)) {
    return false;
  }

  // General case. TODO: We could use a smarter substring algorithm.

  const char* end = haystack->data_ + len(haystack);
  const char* last_possible = end - len(needle);
  const char* p = haystack->data_;

  while (p <= last_possible) {
    if (memcmp(p, needle->data_, len(needle)) == 0) {
      return true;
    }
    p++;
  }
  return false;
}

BigStr* str_repeat(BigStr* s, int times) {
  // Python allows -1 too, and Oil used that
  if (times <= 0) {
    return kEmptyString;
  }
  int len_ = len(s);
  int new_len = len_ * times;
  BigStr* result = NewStr(new_len);

  char* dest = result->data_;
  for (int i = 0; i < times; i++) {
    memcpy(dest, s->data_, len_);
    dest += len_;
  }
  return result;
}

// for os_path.join()
// NOTE(Jesse): Perfect candidate for BoundedBuffer
BigStr* str_concat3(BigStr* a, BigStr* b, BigStr* c) {
  int a_len = len(a);
  int b_len = len(b);
  int c_len = len(c);

  int new_len = a_len + b_len + c_len;
  BigStr* result = NewStr(new_len);
  char* pos = result->data_;

  memcpy(pos, a->data_, a_len);
  pos += a_len;

  memcpy(pos, b->data_, b_len);
  pos += b_len;

  memcpy(pos, c->data_, c_len);

  assert(pos + c_len == result->data_ + new_len);

  return result;
}

BigStr* str_concat(BigStr* a, BigStr* b) {
  int a_len = len(a);
  int b_len = len(b);
  int new_len = a_len + b_len;
  BigStr* result = NewStr(new_len);
  char* buf = result->data_;

  memcpy(buf, a->data_, a_len);
  memcpy(buf + a_len, b->data_, b_len);

  return result;
}

//
// Comparators
//

bool str_equals(BigStr* left, BigStr* right) {
  // Fast path for identical strings.  String deduplication during GC could
  // make this more likely.  String interning could guarantee it, allowing us
  // to remove memcmp().
  if (left == right) {
    return true;
  }

  // TODO: It would be nice to remove this condition, but I think we need MyPy
  // strict None checking for it
  if (left == nullptr || right == nullptr) {
    return false;
  }

  if (left->len_ != right->len_) {
    return false;
  }

  return memcmp(left->data_, right->data_, left->len_) == 0;
}

bool maybe_str_equals(BigStr* left, BigStr* right) {
  if (left && right) {
    return str_equals(left, right);
  }

  if (!left && !right) {
    return true;  // None == None
  }

  return false;  // one is None and one is a BigStr*
}

bool items_equal(BigStr* left, BigStr* right) {
  return str_equals(left, right);
}

bool keys_equal(BigStr* left, BigStr* right) {
  return items_equal(left, right);
}

bool items_equal(Tuple2<int, int>* t1, Tuple2<int, int>* t2) {
  return (t1->at0() == t2->at0()) && (t1->at1() == t2->at1());
}

bool keys_equal(Tuple2<int, int>* t1, Tuple2<int, int>* t2) {
  return items_equal(t1, t2);
}

bool items_equal(Tuple2<BigStr*, int>* t1, Tuple2<BigStr*, int>* t2) {
  return items_equal(t1->at0(), t2->at0()) && (t1->at1() == t2->at1());
}

bool keys_equal(Tuple2<BigStr*, int>* t1, Tuple2<BigStr*, int>* t2) {
  return items_equal(t1, t2);
}

bool str_equals_c(BigStr* s, const char* c_string, int c_len) {
  // Needs SmallStr change
  if (len(s) == c_len) {
    return memcmp(s->data_, c_string, c_len) == 0;
  } else {
    return false;
  }
}

bool str_equals0(const char* c_string, BigStr* s) {
  int n = strlen(c_string);
  if (len(s) == n) {
    return memcmp(s->data_, c_string, n) == 0;
  } else {
    return false;
  }
}

int hash(BigStr* s) {
  return s->hash(fnv1);
}

int max(int a, int b) {
  return std::max(a, b);
}

int min(int a, int b) {
  return std::min(a, b);
}

int max(List<int>* elems) {
  int n = len(elems);
  if (n < 1) {
    throw Alloc<ValueError>();
  }

  int ret = elems->at(0);
  for (int i = 0; i < n; ++i) {
    int cand = elems->at(i);
    if (cand > ret) {
      ret = cand;
    }
  }

  return ret;
}
